#include "asm.h"
#include "dsp32asm.h"
#include "core.h"
#include "format.h"
#include "symbol.h"
#include <stdarg.h>
SRCFILE("dsp32asm.c")

static char buf[128], *sptr;

Dsp32Instr::Dsp32Instr(Asm*a,long l):Instr(a,l)	{ display(); }
char *Dsp32Asm::literaldelimiter()		{ return ""; }
Dsp32Asm::Dsp32Asm(int flag, Core *c):Asm(c)	{ dsp32c = flag; }

Instr *Dsp32Asm::newInstr(long a)
{
        return (Instr*) new Dsp32Instr(this,a);
}

#define	PM_INDIR	(1L<<25)
#define	PM_TOMEM	(1L<<24)
#define	PM_2byte	(1L<<23)

#define	DMOV2TO		(PM_2byte)
#define	DMOV1TO		0L
#define	DMOV2FROM	(PM_TOMEM | PM_2byte)
#define	DMOV1FROM	(PM_TOMEM)
#define	MOV2TO		(PM_INDIR | PM_2byte)
#define	MOV1TO		(PM_INDIR)
#define	MOV2FROM	(PM_INDIR | PM_TOMEM | PM_2byte)
#define	MOV1FROM	(PM_INDIR | PM_TOMEM)

#define DO_OP		0x8C000000
#define IRETURN_INST	0x003E0000
#define PC_REG		0xF
#define	CAU_IOC		27
#define	IOCMASK		0x1FFFFF
#define	LOWBITS(n)		(~(~0L << (n)))
#define	GETBITS(x,p,n)		(((x) >> (p)) & LOWBITS(n))
#define	BITS(p, n)		(LOWBITS(n) << (p))
#define	BITSET(x, bit)		((x) & (1L<<(bit)))
#define	ANY_SET(x,mask)		((x) & (mask))
#define PARITY_BIT_SET(inst)	GETBITS(inst,31,1)
#define	DESTREG(inst)		GETBITS(inst,16,5)
#define	SRCREG(inst)		GETBITS(inst,5,5)
#define	SRCREG2(inst)		GETBITS(inst,0,5)
#define	OPCODE(inst)		GETBITS(inst,21,4)
#define	INDEX(inst)		GETBITS(inst, 0, 16)
#define GET_NE(inst)		GETBITS(inst, 21, 8)
#define	DO_IMMED(inst)		(!GETBITS(inst,21,1))
#define	DO_NUMINST(inst)	GETBITS(inst,16,5)
#define	DO_NUMTIMES(inst)	GETBITS(inst,0,11)
#define DO_REGTIMES(inst)	GETBITS(inst,0,5)
#define	COUNTREG(inst)		GETBITS(inst, 21, 5)
#define	RETURNREG(inst)		GETBITS(inst, 21, 5)
#define IMMEDIATE(inst)		BITSET(inst,25)
#define	K_BIT_SET(inst)		GETBITS(inst,10,1)
#define E_BIT_SET(inst)		GETBITS(inst,11,1)
#define GET_G_BIT(inst)		GETBITS(inst,12,1)
#define GET_CCOND(inst)		GETBITS(inst,13,3)
#define	TSRCREG(inst)		GETBITS(inst, 16, 5)
#define	TDESTREG(inst)		GETBITS(inst, 21, 5)
#define	COND(inst)		GETBITS(inst, 22, 5)
#define	JUMP_RESULT(inst)	(((inst) >> 21) & 1)
#define	GOTO_REG(inst)		GETBITS(inst, 16, 5)
#define	LOBYTBIT(inst)		BITSET(inst, 22)
#define	REG_OP(inst)		GETBITS(inst, 16, 5)
#define	MEM_OP(inst)		GETBITS(inst, 0, 16)
#define	GET_N(inst)		GETBITS(inst, 5, 5)
#define	GET_M(inst)		GETBITS(inst, 0, 5)
#define	WFIELD24	0x3
#define	EXTENDED_MOVE(inst)	(GETBITS(inst,22,2) == WFIELD24)
#define	MREGDIRECT(inst)	(!((inst) & (1L<<21)))
#define	M_ZERO		4
#define	M_ONE		5
#define	M_FORMAT_4	6
#define	REGDIRECT(x)		(!((x) & BITS(3,4)))
#define	RD_IBUF		4
#define	RD_OBUF		5
#define	RD_PDR		6
#define	NOWRITE		7
#define	ZSHIFT		3
#define	YSHIFT		10
#define	XSHIFT		17
#define	YPTRBITS		BITS(YSHIFT, 4)
#define	ZPTRBITS		BITS(ZSHIFT, 4)
#define	Z_PTR(inst)		GETBITS(inst, ZSHIFT, 4)
#define	X_PTR(inst)		GETBITS(inst, XSHIFT, 4)
#define	Y_PTR(inst)		GETBITS(inst, YSHIFT, 4)
#define	Z_CODE(inst)		GETBITS(inst, 0, 7)
#define	Y_CODE(inst)		GETBITS(inst, 7, 7)
#define	X_CODE(inst)		GETBITS(inst, 14, 7)
#define	N_CODE(inst)		GETBITS(inst, 21, 2)
#define	S_CODE(inst)		GETBITS(inst, 23, 1)
#define	F_CODE(inst)		GETBITS(inst, 24, 1)
#define	M_CODE(inst)		GETBITS(inst, 26, 3)
#define	G_CODE(inst)		GETBITS(inst, 23, 4)
#define	DAU_R_BIT(inst)		(!((inst & 0x78000000)==0x78000000) && \
				 GETBITS(inst, 25, 1))
#define	IS_SPECIAL(inst)	((inst & 0xF8000000) == 0x78000000)
#define IS_DOINST(inst)		(((inst) & DO_OP) == DO_OP)
#define	IS_DAU(i)		ANY_SET(i, BITS(29,2))
#define	CAU_BTYPE(i)		GETBITS(i, 26, 3)
#define REV_BIT_7B(x)		(((x) & 0x1E000400L) == 0x1E000400L)

static char res[]= "reserved";

static char	*regstr[32] = {
	"0", "r1", "r2", "r3", "r4", "r5", "r6", "r7",
	"r8", "r9", "r10", "r11", "r12", "fp", "sp", "pc",
	"0", "r15", "r16", "r17", "r18", "r19", "-1", "+1",
	"r20", "r21", "dauc", "ioc", res, "r22", "pcsh", res,
};

static char	*rfield[32] = {
	res, res, res, res, "ibuf", "obuf", "pdr", res,
	res, res, res, res, res, res, "piop", res,
	res, res, res, res, "pdr2", res, "pir", res,
	res, res, res, res, res, res, "pcw", res,
};

static char	*regstra[32] = {
	"0", "r1", "r2", "r3", "r4", "r5", "r6", "r7",
	"r8", "r9", "r10", "r11", "r12", "fp", "sp", "pc",
	res, "r15", "r16", "r17", "r18", "r19", res, res,
	"r20", "r21", res, res, res, "r22", "pcsh", res,
};

static char	*regincstr[32] = {
	"", "++r1", "++r2", "++r3", "++r4", "++r5", "++r6", "++r7",
	"++r8", "++r9", "++r10", "++r11", "++r12", "++fp", "++sp", res,
	"", "++r15", "++r16", "++r17", "++r18", "++r19", "--", "++",
	"++r20", "++r21", res, res, res, "++r22", res, res,
};

static char	*condstr[64] = {
	res, res,
	"pl", "mi",
	"ne", "eq",
	"vc", "vs",
	"cc", "cs",
	"ge", "lt",
	"gt", "le",
	"hi", "ls",

	"auc", "aus",
	"age", "alt",
	"ane", "aeq",
	"avc", "avs",
	"agt", "ale",
	res, res,
	res, res,
	res, res,

	"ibe", "ibf",
	"obf", "obe",
	"pde", "pdf",
	"pie", "pif",
	"syc", "sys",
	"fbc", "fbs",
	"ireq1_lo", "ireq1_hi",
	"ireq2_lo", "ireq2_hi",

	res, res,
	res, res,
	res, res,
	res, res,
	res, res,
	res, res,
	res, res,
	res, res,
};

static	char	esuffix[]	= "e";
static	char	nullsuffix[]	= "";

int Dsp32Instr::dsp32c()	 {return ((Dsp32Asm*)_asm)->dsp32c; }

void Dsp32Instr::p(PRINTF_ARGS)
{
	PRINTF_PROLOGUE;
	sprintf(sptr, PRINTF_COPY);
	sptr = buf + strlen(buf);
}

char *Dsp32Instr::mnemonic()
{
        next = addr+4;
        opword =  _asm->core->peekcode(addr)->lng;
	sptr = buf;
	if (!dsp32c())
		opword &= 0x7fffffffL;
	if (opword == 0)
		p("nop");
	else if (IS_SPECIAL(opword))
		special();
	else switch((int) GETBITS(opword,29,3)) {
		case 1: 
		case 2:
		case 3:
			dau();
			break;
		case 5:	// br24
			p("goto ");
			godest24(1);
			break;
		case 6: // ld24
			m.lng = INDEX(opword) | (GET_NE(opword) << 16 );
			if (m.lng & 0x00800000)	// sign extend
				m.lng |= 0xFF000000; 
			p("%se = %s", regstr[DESTREG(opword)], literal(fmt));
			break;
		case 7:	// call24
			p("call\t");
			godest24(0);
			p("(%s) ", regstr[DESTREG(opword)]);
			break;
		case 4: if (IS_DOINST(opword)) {	// do
				m.lng = DO_NUMINST(opword);
				p("do %s,", literal(fmt));
				if (DO_IMMED(opword)) {
					m.lng = DO_NUMTIMES(opword);
					p("%s", literal(fmt));
				} else
					p("%s", regstr[DO_REGTIMES(opword)]);
				break;
			}
			/* Fall through */
		default:
			switch((int)CAU_BTYPE(opword)) {
			case	0:
			case	1:
				branch();
				break;
			case	2:
				p(res);
				break;
			case	3:	// loop
				p("if(%s-- >= 0) goto ",
					regstr[COUNTREG(opword)]);
				godest();
				break;
			case	4:	// call
				p("call\t");
				godest();
				p("(%s) ", regstr[RETURNREG(opword)]);
				break;
			case	5:
				tripadd();
				break;
			case	6:
				arithl();
				break;
			case	7:
				move();
				break;
			}
			break;
	}
       return sf(buf);
}

void Dsp32Instr::arithl()
{
	static char	*fmtr[16] = {
		"%s%s += %s",
		"%s%s = %s * 2",
		"%s%s = %s - %s",
		"%s%s #= %s",
		"%s%s -= %s",
		"%s%s = - %s",
		"%s%s &~= %s",
		"%s%s - %s",
		"%s%s ^= %s",
		"%s%s = %s >>> 1",
		"%s%s |= %s",
		"%s%s = %s <<< 1",
		"%s%s = %s >> 1",
		"%s%s = %s / 2",
		"%s%s &= %s",
		"%s%s & %s",
	};

	static char *fmtn[] = {
		"%s%s += %s : ILLEGAL",
		"%s%s = %s * 2 : ILLEGAL",
		"%s%s = %s - %s",
		"%s%s #= %s",
		"%s%s -= %s",
		"%s%s = - %s : ILLEGAL",
		"%s%s &~= %s",
		"%s%s - %s",
		"%s%s ^= %s",
		res,
		"%s%s |= %s",
		res,
		"%s%s = %s >> 2 : ILLEGAL",
		"%s%s = %s / 2 : ILLEGAL",
		"%s%s &= %s",
		"%s%s & %s",
	};

	if (dsp32c() && !(IMMEDIATE(opword)) && K_BIT_SET(opword))
		p("if (%s) ", condstr[2*(GET_CCOND(opword)) + GET_G_BIT(opword)]);
	if (dsp32c() && !(IMMEDIATE(opword)) && E_BIT_SET(opword))
		triari();
	else {
		char *suffix = PARITY_BIT_SET(opword) ? esuffix : nullsuffix;
		char *rd = regstr[DESTREG(opword)];
		char *rs = regstr[SRCREG(opword)];
		if(IMMEDIATE(opword)) {
			m.lng = (long)((short)INDEX(opword));
			p(fmtn[OPCODE(opword)], rd, suffix,
			  literal(F_MASKEXT16|fmt), rd);
		} else
			p(fmtr[OPCODE(opword)], rd, suffix, rs, rd);
		p(" ");
	}
}

void Dsp32Instr::triari() // triadic CAU arithmetic and logic
{
	static char	*fmtr[16] = {
		"%s%s = %s + %s",
		res,
		"%s%s = %s - %s",
		"%s%s = %s # %s",
		"%s%s = %s - %s",
		res,
		"%s%s = %s &~ %s",
		res,
		"%s%s = %s ^ %s",
		res,
		"%s%s = %s | %s",
		res,
		res,
		res,
		"%s%s = %s & %s",
		res,
	};

	char *suffix = PARITY_BIT_SET(opword) ? esuffix : nullsuffix;
	char *rd = regstr[DESTREG(opword)];
	char *rs = regstr[SRCREG2(opword)];
	char *rs2= regstr[SRCREG(opword)];
	if (OPCODE(opword) == 2) /* left triadic subtract */
		p(fmtr[OPCODE(opword)], rd, suffix, rs2, rs);
	else
		p(fmtr[OPCODE(opword)], rd, suffix, rs, rs2);
	p(" ");
}

void Dsp32Instr::tripadd()
{
	char *suffix = PARITY_BIT_SET(opword) ? esuffix : nullsuffix;
	int dcode = (int)TDESTREG(opword), scode = (int)TSRCREG(opword);
	char *rd = regstr[dcode];
	char *rs = regstr[scode];
	m.lng = (long)((short)INDEX(opword));
	if (dcode == CAU_IOC) {
		p("%s = ", rd);
		if (dsp32c()) {
			m.lng = opword & IOCMASK;
			p(literal(fmt));
		} else
			p(literal(F_MASKEXT16|fmt));
	}
	else if(scode == 0 || scode == 16)
		// if dest is r18 (ret reg), print symbolic
		p("%s%s = %s", rd, suffix,
			(dcode == 20) ? symbolic() : literal(F_MASKEXT16|fmt));
	else if (m.lng == 0)
		p("%s%s = %s", rd, suffix, rs);
	else {
		char *o = literal(F_MASKEXT16|fmt);
		if (scode == _asm->core->REG_AP() && (fmt & F_SYMBOLIC)) {
			Var *v = local(U_ARG, m.lng);
			if (!v) v = local(U_AUT, m.lng);
			if (v)
				o = v->text();
		}
		p("%s%s = %s + %s", rd, suffix, rs, o);
	}
}

void Dsp32Instr::branch()
{
	int cond = (int)COND(opword);
	if (dsp32c() && opword == IRETURN_INST) {
		p("ireturn\t");
		return;
	}
	if (!cond && !JUMP_RESULT(opword)) {
		p(res);
		return;
	}
	if (cond)
		p("if(%s) ", condstr[cond*2 + (JUMP_RESULT(opword) ? 1:0)]);
	p("goto ");
	godest();
}

// destination address or label in goto and calls
void Dsp32Instr::godest()
{
	if (GOTO_REG(opword) == 0) {
		m.lng = INDEX(opword);
		p("%s ", symbolic());
	} else if (INDEX(opword) == 0 && GOTO_REG(opword) != PC_REG)
		p("%s ", regstr[GOTO_REG(opword)] ) ;
	else if (GOTO_REG(opword) == PC_REG) {
		m.lng = addr+8L+(short)INDEX(opword);
		if (dsp32c())
			m.lng &= 0x00ffffffL;
		else
			m.lng &= 0x0000ffffL;
		p("%s ", symbolic());
	}
	else {
		m.lng = (long)((short)INDEX(opword));
		p("%s + %s ", regstr[GOTO_REG(opword)],
			literal(F_MASKEXT16|fmt));
	}
}

void Dsp32Instr::godest24(char prreg)
{
	m.lng = INDEX(opword) | (GET_NE(opword) << 16);
	if(!prreg || GOTO_REG(opword) == 0)
		p("%s ", symbolic());
	else if(!m.lng && GOTO_REG(opword) != PC_REG)
		p("%s ", regstr[GOTO_REG(opword)] ) ;
	else if (GOTO_REG(opword) == PC_REG) {
		m.lng = (addr+8L+m.lng) & 0x00ffffffL;
		p("%s ", symbolic());
	}
	else {
		if (m.lng & 0x00800000)	// sign extend
			m.lng |= 0xFF000000; 
		p("%s + %s ", regstr[GOTO_REG(opword)], literal(fmt));
	}
}

void Dsp32Instr::move()
{
	register long mcode, ncode;
	register long masked_opword; 
	char *label, *ro, *rm, *rn;

	char *suffix = (EXTENDED_MOVE(opword)) ? "e" : "";
	char *rev_suffix = (dsp32c() && REV_BIT_7B(opword)) ? "r" : "";
	char *bsuff = LOBYTBIT(opword) ? "l" : "h";
	if ((opword & PM_INDIR) || (!(MREGDIRECT(opword)))) {
		ncode = GET_N(opword);
		mcode= GET_M(opword);
		rm = regincstr[mcode];
		rn = regstra[ncode];
	}
	ro = regstra[REG_OP(opword)];
	masked_opword = opword & (PM_INDIR | PM_TOMEM | PM_2byte);
	m.lng = MEM_OP(opword);
	label = symbolic();
	if (masked_opword == DMOV2TO) { 
	  if(!(MREGDIRECT(opword))) 
		p("%s%s = *%s%s", rfield[REG_OP(opword)], suffix, rn, rm);
	  else
		p("%s%s = *%s", ro, suffix, label);
        }
	else if (masked_opword == DMOV1TO) { 
	  if(!(MREGDIRECT(opword))) 
		p("%s%s = *%s%s", rfield[REG_OP(opword)], bsuff, rn, rm);
	  else
		p("%s%s = *%s", ro, bsuff, label);
	}
	else if (masked_opword == DMOV2FROM) { 
	  if(!(MREGDIRECT(opword))) 
		p("*%s%s = %s", rn, rm, rfield[REG_OP(opword)], suffix);
	  else
		p("*%s = %s%s", label, ro, suffix);
 	}
	else if (masked_opword == DMOV1FROM) { 
	  if(!(MREGDIRECT(opword))) 
		p("*%s%s = %s%s", rn, rm, rfield[REG_OP(opword)], bsuff);
	  else
		p("*%s = %s%s", label, ro, bsuff);
	}
	else if (masked_opword == MOV2TO) { 
	  if(MREGDIRECT(opword))
		p("%s%s = %s", ro, suffix, rfield[mcode]);
	  else
		p("%s%s = *%s%s%s", ro, suffix, rn, rm, rev_suffix);
	}
	else if (masked_opword == MOV1TO) { 
	  if(MREGDIRECT(opword))
		p("%s%s = %s", ro, bsuff, rfield[mcode]);
	  else
		p("%s%s = *%s%s%s", ro, bsuff, rn, rm, rev_suffix);
	}
	else if (masked_opword == MOV2FROM) { 
	  if(MREGDIRECT(opword))
		p("%s = %s%s", rfield[mcode], ro, suffix);
	  else
		p("*%s%s%s = %s%s", rn, rm, rev_suffix, ro, suffix);
	}
	else if (masked_opword == MOV1FROM) { 
	  if(MREGDIRECT(opword))
		p("%s = %s%s", rfield[mcode], ro, bsuff);
	  else
		p("*%s%s%s = %s%s", rn, rm, rev_suffix, ro, bsuff);
	} 
}

void Dsp32Instr::putm(unsigned int code)
{
	switch(code) {
	case	M_ZERO:
		p("0.0");
		break;
	case	M_ONE:
		p("1.0");
		break;
	default:
		if(code >= 4)
			p(res);
		else
			p("a%d", code);
	}
}

void Dsp32Instr::putxyz(unsigned int code,char r_bit)
{
	static char	*dauincstr[8] = {
		"", "++r15", "++r16", "++r17", "++r18", "++r19",
		"--", "++"
	};

	if(REGDIRECT(code)) {
		if(code < 4)
			p("a%d", code);
		else if(code == RD_IBUF)
			p("ibuf");
		else if(code == RD_OBUF)
			p("obuf");
		else if(code == RD_PDR)
			p("pdr");
		else
			p(res);
		return;
	}
	int n = (int)GETBITS(code, 3, 4);
	p("*");
	if (n == 14)
		p("sp");
	else if (n == 13)
		p("fp");
	else
		p("r%d", n);
	if(n == 15) p(res);
	p("%s%s", dauincstr[GETBITS(code, 0, 3)], r_bit ? "r" : "");
}

void Dsp32Instr::yzfix()
{
	if(Y_PTR(opword) == 0xf) {
		opword &= ~YPTRBITS;
		opword |= X_PTR(opword) << YSHIFT;
	}
	if(Z_PTR(opword) == 0xf) {
		opword &= ~ZPTRBITS;
		opword |= Y_PTR(opword) << ZSHIFT;
	}
}

void Dsp32Instr::dau()
{
	register unsigned int xcode, ycode, zcode;
	register unsigned int mcode, ncode, fcode, scode;

	yzfix();
	xcode = (unsigned int)X_CODE(opword);
	ycode = (unsigned int)Y_CODE(opword);
	zcode = (unsigned int)Z_CODE(opword);
	mcode = (unsigned int)M_CODE(opword);
	ncode = (unsigned int)N_CODE(opword);
	fcode = (unsigned int)F_CODE(opword);
	scode = (unsigned int)S_CODE(opword);
	char xyzflag = dsp32c() && DAU_R_BIT(opword);

	switch((int)GETBITS(opword, 29, 2)) {
	case	1:	// Z = aN = F(Y) + aM * S(X)
		if (mcode == M_FORMAT_4) {	// aN = F(Z=Y) + S(X)
			p("a%d = %s", ncode, fcode ? "- (" : "(");
			if (zcode != NOWRITE) {
				putxyz(zcode, xyzflag);
				p(" = ");
			}
			putxyz(ycode, 0);
			p(scode ? ") - (" : ") + (");
			putxyz(xcode, 0);
			p(")");
			break;
		}
		if (zcode != NOWRITE) {
			putxyz(zcode, xyzflag);
			p(" = ");
		}
		p("a%d = %s", ncode, fcode ? "-":"");
		putxyz(ycode, 0);
		if (mcode == M_ZERO)
			break;
		p(scode? " - " : " + ");
		if (mcode != M_ONE) {
			putm(mcode);
			p(" * ");
		}
		putxyz(xcode, 0);
		break;
	case	2:	// aN = F(aM) + (Z=Y) * S(X)
		if (mcode == M_ZERO)
			p("a%d = %s", ncode, scode ? "- (" : "(");
		else {
			p("a%d = %s", ncode, fcode? "-":"");
			putm(mcode);
			p(scode? " - (" : " + (" );
		}
		if (zcode != NOWRITE) {
			putxyz(zcode, xyzflag);
			p(" = ");
		}
		putxyz(ycode, 0);
		p(") * ");
		putxyz(xcode, 0);
		break;
	case	3:	// Z = aN = F(aM) + Y*S(X)
		if (zcode != NOWRITE) {
			putxyz(zcode, xyzflag);
			p(" = ");
		}
		if (mcode == M_ZERO)
			p("a%d = %s", ncode, scode ? "- " : "");
		else {
			p("a%d = %s", ncode, fcode? "-":"");
			putm(mcode);
			p(scode? " - " : " + " );
		}
		putxyz(ycode, 0);
		p(" * ");
		putxyz(xcode, 0);
		break;
	default:
		p(res);
	}
}

void Dsp32Instr::special()
{
	static char *funcstr[16] = {
		"ic", "oc", "float", "int", "round", "ifalt",
		"ifaeq", "ifagt",
		res, res, "float24", "int24", "ieee", "dsp",
		"seed", res
	};

	yzfix();
	if (Z_CODE(opword) != NOWRITE) {
		putxyz((unsigned int)Z_CODE(opword), 0);
		p(" = ");
	}
	p("a%d = %s(", (int)N_CODE(opword), funcstr[G_CODE(opword)] );
	putxyz((unsigned int)Y_CODE(opword), 0);
	p(")");
}
